from typing import Literal
import examples.ags.w_action_node.optimized.DROP.graphs.template.operator as operator
import examples.ags.w_action_node.optimized.DROP.graphs.round_21.prompt as prompt_custom
from provider.llm_provider_registry import create_llm_instance
from utils.cost_manager import CostManager

DatasetType = Literal["HumanEval", "MMBP", "Gsm8K", "MATH", "HotpotQa", "MMLU"]

class SolveGraph:
    def __init__(
        self,
        name: str,
        llm_config,
        dataset: DatasetType,
    ) -> None:
        self.name = name
        self.dataset = dataset
        self.llm = create_llm_instance(llm_config)
        self.llm.cost_manager = CostManager()
        self.custom = operator.Custom(self.llm)
        self.answer_generate = operator.AnswerGenerate(self.llm)
        self.sc_ensemble = operator.ScEnsemble(self.llm)

    async def __call__(self, problem: str):
        """
        Implementation of the graph
        """
        step_by_step = await self.answer_generate(input=problem)
        refined_answers = []
        for _ in range(3):  # Generate 3 refined answers
            solution = await self.custom(input=problem + f"\nStep-by-step thought: {step_by_step['thought']}\nInitial answer: {step_by_step['answer']}", instruction=prompt_custom.REFINE_ANSWER_PROMPT)
            refined_answers.append(solution['response'])
        
        ensemble_answer = await self.sc_ensemble(solutions=refined_answers)
        
        final_answer = await self.custom(input=f"Question: {problem}\nInitial answer: {step_by_step['answer']}\nEnsemble answer: {ensemble_answer['response']}", instruction=prompt_custom.FINAL_REFINEMENT_PROMPT)
        
        # Format check and correction
        formatted_answer = await self.custom(input=f"Question: {problem}\nFinal answer: {final_answer['response']}", instruction=prompt_custom.FORMAT_CHECK_PROMPT)
        
        # Error analysis and correction
        error_corrected_answer = await self.custom(input=f"Question: {problem}\nStep-by-step thought: {step_by_step['thought']}\nFormatted answer: {formatted_answer['response']}", instruction=prompt_custom.ERROR_CORRECTION_PROMPT)
        
        # Numerical check and correction
        numerical_corrected_answer = await self.custom(input=f"Question: {problem}\nStep-by-step thought: {step_by_step['thought']}\nCurrent answer: {error_corrected_answer['response']}", instruction=prompt_custom.NUMERICAL_CHECK_PROMPT)
        
        # Range check and correction
        range_corrected_answer = await self.custom(input=f"Question: {problem}\nStep-by-step thought: {step_by_step['thought']}\nCurrent answer: {numerical_corrected_answer['response']}", instruction=prompt_custom.RANGE_CHECK_PROMPT)
        
        # Context-aware answer verification
        context_verified_answer = await self.custom(input=f"Question: {problem}\nStep-by-step thought: {step_by_step['thought']}\nCurrent answer: {range_corrected_answer['response']}", instruction=prompt_custom.CONTEXT_VERIFICATION_PROMPT)
        
        # New step: Numerical precision check and correction
        precision_corrected_answer = await self.custom(input=f"Question: {problem}\nStep-by-step thought: {step_by_step['thought']}\nCurrent answer: {context_verified_answer['response']}", instruction=prompt_custom.PRECISION_CHECK_PROMPT)
        
        return precision_corrected_answer['response'], self.llm.cost_manager.total_cost
                    